/*
# cjt/util/idnDisallowed.js                        Copyright 2022 cPanel, L.L.C.
#                                                           All rights reserved.
# copyright@cpanel.net                                         http://cpanel.net
# This code is subject to the cPanel license. Unauthorized copying is prohibited
*/

/**
 * ----------------------------------------------------------------------
 * idnDisallowed.js - disallowed IDN characters
 *
 * This provides a part of the IDN validation algorithm. It’s not the
 * whole thing—a full IDN validation requires knowledge of Unicode character
 * classes and other such—but it at least gets us part of the way there.
 *
 * The list of characters here comes from RFC 5892 and RFC 6452.
 *
 * Note also that, because of JS limitations, this only examines characters
 * in the Basic Multilingual Plane (U+0000 - U+ffff). (It may be possible to
 * work around this?)
 * ----------------------------------------------------------------------
 *
 * EXAMPLE USAGE:
 *
 * disallowedCharsArray = idnDisallowed.getDisallowedInLabel( labelString )
 *
 * ----------------------------------------------------------------------
 */

define([
    "lodash",
    "punycode",
    "cjt/util/unicode",
], function(_, PUNYCODE, UNICODE) {
    "use strict";

    // A mutation of the table at:
    // https://tools.ietf.org/html/rfc5892#appendix-B.1
    var DISALLOWED = [
        [0x0, 0x2c],
        [0x2e, 0x2f],
        [0x3a, 0x60],
        [0x7b, 0xb6],
        [0xb8, 0xde],
        0xf7,
        0x100,
        0x102,
        0x104,
        0x106,
        0x108,
        0x10a,
        0x10c,
        0x10e,
        0x110,
        0x112,
        0x114,
        0x116,
        0x118,
        0x11a,
        0x11c,
        0x11e,
        0x120,
        0x122,
        0x124,
        0x126,
        0x128,
        0x12a,
        0x12c,
        0x12e,
        0x130,
        [0x132, 0x134],
        0x136,
        0x139,
        0x13b,
        0x13d,
        [0x13f, 0x141],
        0x143,
        0x145,
        0x147,
        [0x149, 0x14a],
        0x14c,
        0x14e,
        0x150,
        0x152,
        0x154,
        0x156,
        0x158,
        0x15a,
        0x15c,
        0x15e,
        0x160,
        0x162,
        0x164,
        0x166,
        0x168,
        0x16a,
        0x16c,
        0x16e,
        0x170,
        0x172,
        0x174,
        0x176,
        [0x178, 0x179],
        0x17b,
        0x17d,
        0x17f,
        [0x181, 0x182],
        0x184,
        [0x186, 0x187],
        [0x189, 0x18b],
        [0x18e, 0x191],
        [0x193, 0x194],
        [0x196, 0x198],
        [0x19c, 0x19d],
        [0x19f, 0x1a0],
        0x1a2,
        0x1a4,
        [0x1a6, 0x1a7],
        0x1a9,
        0x1ac,
        [0x1ae, 0x1af],
        [0x1b1, 0x1b3],
        0x1b5,
        [0x1b7, 0x1b8],
        0x1bc,
        [0x1c4, 0x1cd],
        0x1cf,
        0x1d1,
        0x1d3,
        0x1d5,
        0x1d7,
        0x1d9,
        0x1db,
        0x1de,
        0x1e0,
        0x1e2,
        0x1e4,
        0x1e6,
        0x1e8,
        0x1ea,
        0x1ec,
        0x1ee,
        [0x1f1, 0x1f4],
        [0x1f6, 0x1f8],
        0x1fa,
        0x1fc,
        0x1fe,
        0x200,
        0x202,
        0x204,
        0x206,
        0x208,
        0x20a,
        0x20c,
        0x20e,
        0x210,
        0x212,
        0x214,
        0x216,
        0x218,
        0x21a,
        0x21c,
        0x21e,
        0x220,
        0x222,
        0x224,
        0x226,
        0x228,
        0x22a,
        0x22c,
        0x22e,
        0x230,
        0x232,
        [0x23a, 0x23b],
        [0x23d, 0x23e],
        0x241,
        [0x243, 0x246],
        0x248,
        0x24a,
        0x24c,
        0x24e,
        [0x2b0, 0x2b8],
        [0x2c2, 0x2c5],
        [0x2d2, 0x2eb],
        0x2ed,
        [0x2ef, 0x2ff],
        [0x340, 0x341],
        [0x343, 0x345],
        0x34f,
        0x370,
        0x372,
        0x374,
        0x376,
        0x37a,
        0x37e,
        [0x384, 0x38a],
        0x38c,
        [0x38e, 0x38f],
        [0x391, 0x3a1],
        [0x3a3, 0x3ab],
        [0x3cf, 0x3d6],
        0x3d8,
        0x3da,
        0x3dc,
        0x3de,
        0x3e0,
        0x3e2,
        0x3e4,
        0x3e6,
        0x3e8,
        0x3ea,
        0x3ec,
        0x3ee,
        [0x3f0, 0x3f2],
        [0x3f4, 0x3f7],
        [0x3f9, 0x3fa],
        [0x3fd, 0x42f],
        0x460,
        0x462,
        0x464,
        0x466,
        0x468,
        0x46a,
        0x46c,
        0x46e,
        0x470,
        0x472,
        0x474,
        0x476,
        0x478,
        0x47a,
        0x47c,
        0x47e,
        0x480,
        0x482,
        [0x488, 0x48a],
        0x48c,
        0x48e,
        0x490,
        0x492,
        0x494,
        0x496,
        0x498,
        0x49a,
        0x49c,
        0x49e,
        0x4a0,
        0x4a2,
        0x4a4,
        0x4a6,
        0x4a8,
        0x4aa,
        0x4ac,
        0x4ae,
        0x4b0,
        0x4b2,
        0x4b4,
        0x4b6,
        0x4b8,
        0x4ba,
        0x4bc,
        0x4be,
        [0x4c0, 0x4c1],
        0x4c3,
        0x4c5,
        0x4c7,
        0x4c9,
        0x4cb,
        0x4cd,
        0x4d0,
        0x4d2,
        0x4d4,
        0x4d6,
        0x4d8,
        0x4da,
        0x4dc,
        0x4de,
        0x4e0,
        0x4e2,
        0x4e4,
        0x4e6,
        0x4e8,
        0x4ea,
        0x4ec,
        0x4ee,
        0x4f0,
        0x4f2,
        0x4f4,
        0x4f6,
        0x4f8,
        0x4fa,
        0x4fc,
        0x4fe,
        0x500,
        0x502,
        0x504,
        0x506,
        0x508,
        0x50a,
        0x50c,
        0x50e,
        0x510,
        0x512,
        0x514,
        0x516,
        0x518,
        0x51a,
        0x51c,
        0x51e,
        0x520,
        0x522,
        0x524,
        [0x531, 0x556],
        [0x55a, 0x55f],
        0x587,
        [0x589, 0x58a],
        0x5be,
        0x5c0,
        0x5c3,
        0x5c6,
        [0x600, 0x603],
        [0x606, 0x60f],
        0x61b,
        [0x61e, 0x61f],
        0x640,
        [0x66a, 0x66d],
        [0x675, 0x678],
        0x6d4,
        [0x6dd, 0x6de],
        0x6e9,
        [0x700, 0x70d],
        0x70f,
        [0x7f6, 0x7fa],
        [0x830, 0x83e],
        [0x958, 0x95f],
        [0x964, 0x965],
        0x970,
        [0x9dc, 0x9dd],
        0x9df,
        [0x9f2, 0x9fb],
        0xa33,
        0xa36,
        [0xa59, 0xa5b],
        0xa5e,
        0xaf1,
        [0xb5c, 0xb5d],
        0xb70,
        [0xbf0, 0xbfa],
        [0xc78, 0xc7f],

        // [0xcf1, 0xcf2],  RFC 6452 allows this now
        [0xd70, 0xd75],
        0xd79,
        0xdf4,
        0xe33,
        0xe3f,
        0xe4f,
        [0xe5a, 0xe5b],
        0xeb3,
        [0xedc, 0xedd],
        [0xf01, 0xf0a],
        [0xf0c, 0xf17],
        [0xf1a, 0xf1f],
        [0xf2a, 0xf34],
        0xf36,
        0xf38,
        [0xf3a, 0xf3d],
        0xf43,
        0xf4d,
        0xf52,
        0xf57,
        0xf5c,
        0xf69,
        0xf73,
        [0xf75, 0xf79],
        0xf81,
        0xf85,
        0xf93,
        0xf9d,
        0xfa2,
        0xfa7,
        0xfac,
        0xfb9,
        [0xfbe, 0xfc5],
        [0xfc7, 0xfcc],
        [0xfce, 0xfd8],
        [0x104a, 0x104f],
        [0x109e, 0x10c5],
        [0x10fb, 0x10fc],
        [0x1100, 0x11ff],
        [0x1360, 0x137c],
        [0x1390, 0x1399],
        0x1400,
        [0x166d, 0x166e],
        0x1680,
        [0x169b, 0x169c],
        [0x16eb, 0x16f0],
        [0x1735, 0x1736],
        [0x17b4, 0x17b5],
        [0x17d4, 0x17d6],
        [0x17d8, 0x17db],
        [0x17f0, 0x17f9],
        [0x1800, 0x180e],
        0x1940,
        [0x1944, 0x1945],
        [0x19de, 0x19ff],
        [0x1a1e, 0x1a1f],
        [0x1aa0, 0x1aa6],
        [0x1aa8, 0x1aad],
        [0x1b5a, 0x1b6a],
        [0x1b74, 0x1b7c],
        [0x1c3b, 0x1c3f],
        [0x1c7e, 0x1c7f],
        0x1cd3,
        [0x1d2c, 0x1d2e],
        [0x1d30, 0x1d3a],
        [0x1d3c, 0x1d4d],
        [0x1d4f, 0x1d6a],
        0x1d78,
        [0x1d9b, 0x1dbf],
        0x1e00,
        0x1e02,
        0x1e04,
        0x1e06,
        0x1e08,
        0x1e0a,
        0x1e0c,
        0x1e0e,
        0x1e10,
        0x1e12,
        0x1e14,
        0x1e16,
        0x1e18,
        0x1e1a,
        0x1e1c,
        0x1e1e,
        0x1e20,
        0x1e22,
        0x1e24,
        0x1e26,
        0x1e28,
        0x1e2a,
        0x1e2c,
        0x1e2e,
        0x1e30,
        0x1e32,
        0x1e34,
        0x1e36,
        0x1e38,
        0x1e3a,
        0x1e3c,
        0x1e3e,
        0x1e40,
        0x1e42,
        0x1e44,
        0x1e46,
        0x1e48,
        0x1e4a,
        0x1e4c,
        0x1e4e,
        0x1e50,
        0x1e52,
        0x1e54,
        0x1e56,
        0x1e58,
        0x1e5a,
        0x1e5c,
        0x1e5e,
        0x1e60,
        0x1e62,
        0x1e64,
        0x1e66,
        0x1e68,
        0x1e6a,
        0x1e6c,
        0x1e6e,
        0x1e70,
        0x1e72,
        0x1e74,
        0x1e76,
        0x1e78,
        0x1e7a,
        0x1e7c,
        0x1e7e,
        0x1e80,
        0x1e82,
        0x1e84,
        0x1e86,
        0x1e88,
        0x1e8a,
        0x1e8c,
        0x1e8e,
        0x1e90,
        0x1e92,
        0x1e94,
        [0x1e9a, 0x1e9b],
        0x1e9e,
        0x1ea0,
        0x1ea2,
        0x1ea4,
        0x1ea6,
        0x1ea8,
        0x1eaa,
        0x1eac,
        0x1eae,
        0x1eb0,
        0x1eb2,
        0x1eb4,
        0x1eb6,
        0x1eb8,
        0x1eba,
        0x1ebc,
        0x1ebe,
        0x1ec0,
        0x1ec2,
        0x1ec4,
        0x1ec6,
        0x1ec8,
        0x1eca,
        0x1ecc,
        0x1ece,
        0x1ed0,
        0x1ed2,
        0x1ed4,
        0x1ed6,
        0x1ed8,
        0x1eda,
        0x1edc,
        0x1ede,
        0x1ee0,
        0x1ee2,
        0x1ee4,
        0x1ee6,
        0x1ee8,
        0x1eea,
        0x1eec,
        0x1eee,
        0x1ef0,
        0x1ef2,
        0x1ef4,
        0x1ef6,
        0x1ef8,
        0x1efa,
        0x1efc,
        0x1efe,
        [0x1f08, 0x1f0f],
        [0x1f18, 0x1f1d],
        [0x1f28, 0x1f2f],
        [0x1f38, 0x1f3f],
        [0x1f48, 0x1f4d],
        0x1f59,
        0x1f5b,
        0x1f5d,
        0x1f5f,
        [0x1f68, 0x1f6f],
        0x1f71,
        0x1f73,
        0x1f75,
        0x1f77,
        0x1f79,
        0x1f7b,
        0x1f7d,
        [0x1f80, 0x1faf],
        [0x1fb2, 0x1fb4],
        [0x1fb7, 0x1fc4],
        [0x1fc7, 0x1fcf],
        0x1fd3,
        [0x1fd8, 0x1fdb],
        [0x1fdd, 0x1fdf],
        0x1fe3,
        [0x1fe8, 0x1fef],
        [0x1ff2, 0x1ff4],
        [0x1ff7, 0x1ffe],
        [0x2000, 0x200b],
        [0x200e, 0x2064],
        [0x206a, 0x2071],
        [0x2074, 0x208e],
        [0x2090, 0x2094],
        [0x20a0, 0x20b8],
        [0x20d0, 0x20f0],
        [0x2100, 0x214d],
        [0x214f, 0x2183],
        [0x2185, 0x2189],
        [0x2190, 0x23e8],
        [0x2400, 0x2426],
        [0x2440, 0x244a],
        [0x2460, 0x26cd],
        [0x26cf, 0x26e1],
        0x26e3,
        [0x26e8, 0x26ff],
        [0x2701, 0x2704],
        [0x2706, 0x2709],
        [0x270c, 0x2727],
        [0x2729, 0x274b],
        0x274d,
        [0x274f, 0x2752],
        [0x2756, 0x275e],
        [0x2761, 0x2794],
        [0x2798, 0x27af],
        [0x27b1, 0x27be],
        [0x27c0, 0x27ca],
        0x27cc,
        [0x27d0, 0x2b4c],
        [0x2b50, 0x2b59],
        [0x2c00, 0x2c2e],
        0x2c60,
        [0x2c62, 0x2c64],
        0x2c67,
        0x2c69,
        0x2c6b,
        [0x2c6d, 0x2c70],
        0x2c72,
        0x2c75,
        [0x2c7c, 0x2c80],
        0x2c82,
        0x2c84,
        0x2c86,
        0x2c88,
        0x2c8a,
        0x2c8c,
        0x2c8e,
        0x2c90,
        0x2c92,
        0x2c94,
        0x2c96,
        0x2c98,
        0x2c9a,
        0x2c9c,
        0x2c9e,
        0x2ca0,
        0x2ca2,
        0x2ca4,
        0x2ca6,
        0x2ca8,
        0x2caa,
        0x2cac,
        0x2cae,
        0x2cb0,
        0x2cb2,
        0x2cb4,
        0x2cb6,
        0x2cb8,
        0x2cba,
        0x2cbc,
        0x2cbe,
        0x2cc0,
        0x2cc2,
        0x2cc4,
        0x2cc6,
        0x2cc8,
        0x2cca,
        0x2ccc,
        0x2cce,
        0x2cd0,
        0x2cd2,
        0x2cd4,
        0x2cd6,
        0x2cd8,
        0x2cda,
        0x2cdc,
        0x2cde,
        0x2ce0,
        0x2ce2,
        [0x2ce5, 0x2ceb],
        0x2ced,
        [0x2cf9, 0x2cff],
        0x2d6f,
        [0x2e00, 0x2e2e],
        [0x2e30, 0x2e31],
        [0x2e80, 0x2e99],
        [0x2e9b, 0x2ef3],
        [0x2f00, 0x2fd5],
        [0x2ff0, 0x2ffb],
        [0x3000, 0x3004],
        [0x3008, 0x3029],
        [0x302e, 0x303b],
        [0x303d, 0x303f],
        [0x309b, 0x309c],
        [0x309f, 0x30a0],
        0x30ff,
        [0x3131, 0x318e],
        [0x3190, 0x319f],
        [0x31c0, 0x31e3],
        [0x3200, 0x321e],
        [0x3220, 0x32fe],
        [0x3300, 0x33ff],
        [0x4dc0, 0x4dff],
        [0xa490, 0xa4c6],
        [0xa4fe, 0xa4ff],
        [0xa60d, 0xa60f],
        0xa640,
        0xa642,
        0xa644,
        0xa646,
        0xa648,
        0xa64a,
        0xa64c,
        0xa64e,
        0xa650,
        0xa652,
        0xa654,
        0xa656,
        0xa658,
        0xa65a,
        0xa65c,
        0xa65e,
        0xa662,
        0xa664,
        0xa666,
        0xa668,
        0xa66a,
        0xa66c,
        [0xa670, 0xa673],
        0xa67e,
        0xa680,
        0xa682,
        0xa684,
        0xa686,
        0xa688,
        0xa68a,
        0xa68c,
        0xa68e,
        0xa690,
        0xa692,
        0xa694,
        0xa696,
        [0xa6e6, 0xa6ef],
        [0xa6f2, 0xa6f7],
        [0xa700, 0xa716],
        [0xa720, 0xa722],
        0xa724,
        0xa726,
        0xa728,
        0xa72a,
        0xa72c,
        0xa72e,
        0xa732,
        0xa734,
        0xa736,
        0xa738,
        0xa73a,
        0xa73c,
        0xa73e,
        0xa740,
        0xa742,
        0xa744,
        0xa746,
        0xa748,
        0xa74a,
        0xa74c,
        0xa74e,
        0xa750,
        0xa752,
        0xa754,
        0xa756,
        0xa758,
        0xa75a,
        0xa75c,
        0xa75e,
        0xa760,
        0xa762,
        0xa764,
        0xa766,
        0xa768,
        0xa76a,
        0xa76c,
        0xa76e,
        0xa770,
        0xa779,
        0xa77b,
        [0xa77d, 0xa77e],
        0xa780,
        0xa782,
        0xa784,
        0xa786,
        [0xa789, 0xa78b],
        [0xa828, 0xa82b],
        [0xa830, 0xa839],
        [0xa874, 0xa877],
        [0xa8ce, 0xa8cf],
        [0xa8f8, 0xa8fa],
        [0xa92e, 0xa92f],
        [0xa95f, 0xa97c],
        [0xa9c1, 0xa9cd],
        [0xa9de, 0xa9df],
        [0xaa5c, 0xaa5f],
        [0xaa77, 0xaa79],
        [0xaade, 0xaadf],
        0xabeb,
        [0xd7b0, 0xd7c6],
        [0xd7cb, 0xd7fb],
        [0xd800, 0xfa0d],
        0xfa10,
        0xfa12,
        [0xfa15, 0xfa1e],
        0xfa20,
        0xfa22,
        [0xfa25, 0xfa26],
        [0xfa2a, 0xfa2d],
        [0xfa30, 0xfa6d],
        [0xfa70, 0xfad9],
        [0xfb00, 0xfb06],
        [0xfb13, 0xfb17],
        0xfb1d,
        [0xfb1f, 0xfb36],
        [0xfb38, 0xfb3c],
        0xfb3e,
        [0xfb40, 0xfb41],
        [0xfb43, 0xfb44],
        [0xfb46, 0xfbb1],
        [0xfbd3, 0xfd3f],
        [0xfd50, 0xfd8f],
        [0xfd92, 0xfdc7],
        [0xfdd0, 0xfdfd],
        [0xfe00, 0xfe19],
        [0xfe30, 0xfe52],
        [0xfe54, 0xfe66],
        [0xfe68, 0xfe6b],
        [0xfe70, 0xfe72],
        0xfe74,
        [0xfe76, 0xfefc],
        0xfeff,
        [0xff01, 0xffbe],
        [0xffc2, 0xffc7],
        [0xffca, 0xffcf],
        [0xffd2, 0xffd7],
        [0xffda, 0xffdc],
        [0xffe0, 0xffe6],
        [0xffe8, 0xffee],
        [0xfff9, 0xffff],

        /*
        ** Code points above 0xffff cause JS (PhantomJS, anyway)
        ** to misidentify ordinary letters like “f” as disallowed.
        ** punycode.js’s ucs2 stuff compensates for that.
        */
        [0x10100, 0x10102],
        [0x10107, 0x10133],
        [0x10137, 0x1018a],
        [0x10190, 0x1019b],
        [0x101d0, 0x101fc],
        [0x10320, 0x10323],
        0x10341,
        0x1034a,
        0x1039f,
        [0x103d0, 0x103d5],
        [0x10400, 0x10427],
        [0x10857, 0x1085f],
        [0x10916, 0x1091b],
        0x1091f,
        0x1093f,
        [0x10a40, 0x10a47],
        [0x10a50, 0x10a58],
        [0x10a7d, 0x10a7f],
        [0x10b39, 0x10b3f],
        [0x10b58, 0x10b5f],
        [0x10b78, 0x10b7f],
        [0x10e60, 0x10e7e],
        [0x110bb, 0x110c1],
        [0x12400, 0x12462],
        [0x12470, 0x12473],
        [0x1d000, 0x1d0f5],
        [0x1d100, 0x1d126],
        [0x1d129, 0x1d1dd],
        [0x1d200, 0x1d245],
        [0x1d300, 0x1d356],
        [0x1d360, 0x1d371],
        [0x1d400, 0x1d454],
        [0x1d456, 0x1d49c],
        [0x1d49e, 0x1d49f],
        0x1d4a2,
        [0x1d4a5, 0x1d4a6],
        [0x1d4a9, 0x1d4ac],
        [0x1d4ae, 0x1d4b9],
        0x1d4bb,
        [0x1d4bd, 0x1d4c3],
        [0x1d4c5, 0x1d505],
        [0x1d507, 0x1d50a],
        [0x1d50d, 0x1d514],
        [0x1d516, 0x1d51c],
        [0x1d51e, 0x1d539],
        [0x1d53b, 0x1d53e],
        [0x1d540, 0x1d544],
        0x1d546,
        [0x1d54a, 0x1d550],
        [0x1d552, 0x1d6a5],
        [0x1d6a8, 0x1d7cb],
        [0x1d7ce, 0x1d7ff],
        [0x1f000, 0x1f02b],
        [0x1f030, 0x1f093],
        [0x1f100, 0x1f10a],
        [0x1f110, 0x1f12e],
        0x1f131,
        0x1f13d,
        0x1f13f,
        0x1f142,
        0x1f146,
        [0x1f14a, 0x1f14e],
        0x1f157,
        0x1f15f,
        0x1f179,
        [0x1f17b, 0x1f17c],
        0x1f17f,
        [0x1f18a, 0x1f18d],
        0x1f190,
        0x1f200,
        [0x1f210, 0x1f231],
        [0x1f240, 0x1f248],
        [0x1fffe, 0x1ffff],
        [0x2f800, 0x2fa1d],
        [0x2fffe, 0x2ffff],
        [0x3fffe, 0x3ffff],
        [0x4fffe, 0x4ffff],
        [0x5fffe, 0x5ffff],
        [0x6fffe, 0x6ffff],
        [0x7fffe, 0x7ffff],
        [0x8fffe, 0x8ffff],
        [0x9fffe, 0x9ffff],
        [0xafffe, 0xaffff],
        [0xbfffe, 0xbffff],
        [0xcfffe, 0xcffff],
        [0xdfffe, 0xdffff],
        0xe0001,
        [0xe0020, 0xe007f],
        [0xe0100, 0xe01ef],
        [0xefffe, 0x10ffff],

        0x19da, // RFC 6452 addition
    ];

    // A transform of the above datastore to a lookup of single characters.
    var DISALLOWED_LOOKUP;

    function _getDisallowedLookup() {
        if (!DISALLOWED_LOOKUP) {
            DISALLOWED_LOOKUP = UNICODE.createCharacterLookup(DISALLOWED);
        }

        return DISALLOWED_LOOKUP;
    }

    /**
     * @function getDisallowedInLabel
     * @param String specimen - the input
     * @return Array an array of the unique invalid characters from the input
     */
    function getDisallowedInLabel(specimen) {

        var disallowedLookup = _getDisallowedLookup();

        var codePoints = PUNYCODE.ucs2.decode(specimen);

        // We could use an object/dict for this, but we’d depend on
        // object insertion order being preserved in iteration.
        var badChars = [];

        var myChar;

        for (var i = 0; i < codePoints.length; i++) {
            myChar = (codePoints[i] > 0xffff) ? PUNYCODE.ucs2.encode([codePoints[i]]) : String.fromCharCode(codePoints[i]);

            if ( disallowedLookup[myChar] ) {
                badChars.push(myChar);
            }
        }

        return _.uniq(badChars);
    }

    return {
        getDisallowedInLabel: getDisallowedInLabel,
    };
} );
